import math

def line_intersection(a1, b1, c1, a2, b2, c2, tol=1e-8):
    """Compute the intersection of two lines a1*x + b1*y = c1 and a2*x + b2*y = c2.
       Returns (x, y) if they intersect, or None if they are parallel."""
    det = a1 * b2 - a2 * b1
    if abs(det) < tol:
        return None
    x = (c1 * b2 - c2 * b1) / det
    y = (a1 * c2 - a2 * c1) / det
    return (x, y)

def satisfies_all(x, y, constraints, tol=1e-8):
    """Check if the point (x, y) satisfies all inequalities a*x + b*y <= c in the given constraints."""
    return all(a*x + b*y <= c + tol for a, b, c in constraints)

def get_full_constraints(A, b_vec):
    """Return the list of constraints from matrix A and b vector with non-negativity added."""
    constraints = [(row[0], row[1], b) for row, b in zip(A, b_vec)]
    # Add non-negativity: x >= 0 and y >= 0 written as -x <= 0 and -y <= 0.
    constraints.append((-1, 0, 0))
    constraints.append((0, -1, 0))
    return constraints

def get_tangent_vector(constraint, p, constraints, tol=1e-8, eps=1e-2):
    """
    For a given constraint (a, b, c) and an intersection point p,
    return a tangent vector along the boundary.
    A candidate is (-b, a) and is reversed if a small step along it violates feasibility.
    """
    a, b, _ = constraint
    candidate = (-b, a)
    test_pt = (p[0] + eps * candidate[0], p[1] + eps * candidate[1])
    if satisfies_all(test_pt[0], test_pt[1], constraints, tol):
        return candidate
    return (-candidate[0], -candidate[1])

def compute_chosen_intersection(A, b_vec, i_idx, j_idx):
    """Compute the intersection point of the two chosen constraints from A and b_vec."""
    a1, b1 = A[i_idx]
    a2, b2 = A[j_idx]
    c1 = b_vec[i_idx]
    c2 = b_vec[j_idx]
    return line_intersection(a1, b1, c1, a2, b2, c2)

def tangent_vectors(A, b_vec, i_idx, j_idx):
    """
    Compute the tangent (edge) vectors at the intersection of the two chosen constraints.
    """
    constraints = get_full_constraints(A, b_vec)
    inter_pt = compute_chosen_intersection(A, b_vec, i_idx, j_idx)
    if inter_pt is None or not satisfies_all(inter_pt[0], inter_pt[1], constraints):
        print("Chosen intersection is not feasible.")
        return

    # Get tangent vectors for each chosen constraint.
    constraint_i = (A[i_idx][0], A[i_idx][1], b_vec[i_idx])
    tangent_i = get_tangent_vector(constraint_i, inter_pt, constraints)
    constraint_j = (A[j_idx][0], A[j_idx][1], b_vec[j_idx])
    tangent_j = get_tangent_vector(constraint_j, inter_pt, constraints)

    return inter_pt, tangent_i, tangent_j

def is_feasible_intersection(b_vec, A, i_idx, j_idx, tol=1e-8):
    """Check whether the intersection of the two chosen constraints is feasible."""
    n1 = A[i_idx]
    n2 = A[j_idx]
    d1, d2 = b_vec[i_idx], b_vec[j_idx]
    det = n1[0] * n2[1] - n1[1] * n2[0]
    if abs(det) < tol:
        return False
    x_inter = (d1 * n2[1] - d2 * n1[1]) / det
    y_inter = (n1[0] * d2 - n2[0] * d1) / det
    if x_inter < 0 or y_inter < 0:
        return False
    for idx, (a, b) in enumerate(A):
        if a * x_inter + b * y_inter > b_vec[idx] + tol:
            return False
    return True

def solution(A, b_vec, i_idx, j_idx, c):
        # Check feasibility of the chosen intersection.
    if not is_feasible_intersection(b_vec, A, i_idx, j_idx):
        return 0

    _, tangent_i, tangent_j = tangent_vectors(A, b_vec, i_idx, j_idx)

    p1 = tangent_i[0] * c[0] + tangent_i[1] * c[1]
    p2 = tangent_j[0] * c[0] + tangent_j[1] * c[1]
    
    if p1 < 0 and p2 < 0:
        return 1
    else: 
        return 0

if __name__ == "__main__":
    c = list(map(float, input().split()))
    n = int(input())
    b_vec = list(map(float, input().split()))
    A = [list(map(float, input().split())) for _ in range(n)]
    indices = list(map(int, input().split()))
    i_idx, j_idx = indices[0] - 1, indices[1] - 1

    answer = solution(A, b_vec, i_idx, j_idx, c)
    print(answer)